// StupidExperimenterPlugin.cpp

#include <OS.h>
#include <stdio.h>
#include "EnvironmentConstants.h"
#include "StupidExperimenterPlugin.h"
#include "Debug.h"
#include "UserEnvMessage.h"
#include "PluginSettingsNames.h"
#include "DatabaseMessage.h"
#include "DatabaseFieldNames.h"

ExperimenterPlugin *instantiate_exp(PortMessage *expRatPort,
	PortMessage *envExpPort, DatabaseMessage *dbMsg, UserEnvMessage *userEnvMsg,
	DebugServer *bugServer) {
   return new StupidExperimenterPlugin(expRatPort, envExpPort, dbMsg, userEnvMsg,
   	bugServer);
}

/*******************************************************
* Created: 8/30/00, CGP
* 	Stupid experimenter.
* Modified: 9/15/00, CGP
* 		Enabled default plugin settings.
* Modified: 9/19/00, CGP
*		Revised to enable variable number of arms.
* Modified: 9/20/00, CGP
* 		Revised to stop the rat when all food gone from maze.
* Modified: 10/12/00, CGP
*		Added test inserts of performance data records into database, and test query
* Modified: 11/4/00, CGP
*		To use unified experimenter API.
* Modified: 12/24/00, CGP
*		The rat API has changed so that the rat no longer uses an explicit message
*		interface. After we send a STOP message to the rat, we need
*		to place the rat where we want him next (e.g., in the center of the maze).
* Modified: 1/3/01, CGP
*		Just tracked down a difficult, deadlock, bug. The symptoms were that the
*		maze simulator was crashing after the rat had run awhile.
*		It turned out to be crashing trying to send a message to the environment.
*		What was happening was that the experimenter was trying to send a message
*		to the environment (GetCurrHeading()) and the environment was simulataneously
*		trying to send a message to the experimenter. This means that if we give the
*		programmer of the experimenters the possibility of explicitly sending messages
*		to the environment, we may deadlock. More generally, since we have the
*		environment sending messages to the experimenter and the experimenter
*		sending messages to the environment, we must make some change so that we avoid
*		deadlock. It seems like the best way to do this right now is to change the
*		messaging from the environment to the experimenter so that messages sent from
*		the environment to the experimenter do not block the environment. Since
*		the environment is just sending informational messages to the experimenter
*		it does not have to receive a reply from the experimenter, therefore, the
*		environment does not need to block when it sends a message to the experimenter.
*		Even though the environment was sending messages to the experimenter via a port
*		it appears that the port was filling and causing the environment to block.
*		This was resolved by providing unbounded (based on memory constraints) message
*		buffering for the messages sent from the environment to the experimenter.
* Modified: 1/5/01, CGP
*		The Experimenter API is now event driven. The ExperimenterPlugin superclass
*		provides a thread that runs, receives messages, and calls the relevant
*		functions of the experimenter: Setup, RunTrial, RatConsumedFood, RatMoved etc.
*
*******************************************************/

StupidExperimenterPlugin::StupidExperimenterPlugin(PortMessage *expRatPort,
	PortMessage *envExpPort, DatabaseMessage *dbMsg, UserEnvMessage *userEnvMsg,
	DebugServer *bugServer):
		ExperimenterPlugin(expRatPort, envExpPort, dbMsg, userEnvMsg, bugServer)
{
	// temporary
	settings = new BMessage();
	
	// create default plugin settings, if none given
	if (settings->CountNames(B_ANY_TYPE) == 0) {
		Debug('n', "StupidExperimenterPlugin::StupidExperimenterPlugin: Null default settings");
		this->settings = new BMessage();
		this->settings->AddInt16(NUMBER_OF_MAZE_ARMS, 8);
	} // default constructor already established value for this->settings otherwise
}

StupidExperimenterPlugin::~StupidExperimenterPlugin()
{
}

// Setup: Open the maze doors & initialize the rat number
void StupidExperimenterPlugin::PLUGIN_Setup()
{
	char buf[50];
	
	Debug('t', "StupidExperimenterPlugin::Setup");

	// Assuming for now that the number of maze arms doesn't change once
	// system is started...
	numArms = EXP_GetNumberOfMazeArms();
	for (int arm=0; arm < numArms; arm++) {
		EXP_OpenDoor(arm);
	}

	int heading = EXP_GetRatHeading();
	sprintf(buf, "%d", heading);
	
	Debug('n', "StupidExperimenterPlugin: Started... Rat headed (degrees): ", buf);

	ratNo = 1;
	EXP_SetCurrentRatNumber(ratNo);
}



void StupidExperimenterPlugin::BaitMaze()
{
	for (int arm=0; arm < numArms; arm++) {
		EXP_PlaceFood(arm);
	}
	
	EXP_PlaceRatAt(CENTER);
	numFood = numArms;
}

void StupidExperimenterPlugin::PLUGIN_RunTrial()
{
	Debug('t', "StupidExperimenterPlugin::RunTrial");
	
	EXP_SetCurrentRatNumber(ratNo);
	BaitMaze();
			
	Debug('n', "StupidExperimenterPlugin::RunTrial: Putting rat on maze");
	EXP_PutRatOnMaze();
}

/*
// Assume that GUI is in step mode. Just run one step for rat.
// Our rat just goes on the maze, looking for food. So taking a single step
// only makes sense if there is food on the maze. Let's check if there is food
// on the maze before running a single step. If there is no food, then bait the entire
// maze.
void StupidExperimenterPlugin::PLUGIN_RunStep()
{	
	if (! EXP_AnyFoodOnMaze()) {
		BaitMaze();
	}
	
	EXP_SingleStepRat();
}
*/

void StupidExperimenterPlugin::PLUGIN_RatMoved(BMessage *msg)
{
	BMessage *perfDataRecord;
	int16 armNumber, foodFlag;
	
	Debug('t', "StupidExperimenterPlugin::RatMoved");
	
	// message for telling us that rat moved also contains "armNumber" field
	if (msg->FindInt16("armNumber", &armNumber) != B_OK) {
		Debug('e', "StupidExperimenterPlugin::RatMoved: No armNumber field");
		armNumber = -999;
	}
	
	// If rat moves to an arm, the message that we get sent also
	// tells us whether or not the arm has food
	if (armNumber != CENTER) {
		if (msg->FindInt16("foodFlag", &foodFlag) != B_OK) {
			Debug('e', "StupidExperimenterPlugin::RatMoved: No foodFlag field");
			foodFlag = -1;
		}
		
		perfDataRecord = new BMessage();
		// Database only allows int16's right now as data type
		// Record type 2 is for the StupidExperimenter
		perfDataRecord->AddInt16(DBFLD_RECORD_TYPE, 2);
		perfDataRecord->AddInt16(DBFLD_RAT_NUMBER, ratNo);
		perfDataRecord->AddInt16(DBFLD_ARM_NUMBER, armNumber);
		perfDataRecord->AddInt16(DBFLD_CONTAINED_FOOD, foodFlag);				
		EXP_AddDatabaseRecord(perfDataRecord);
		delete perfDataRecord;
	}
}

void StupidExperimenterPlugin::PLUGIN_RatConsumedFood()
{
	char buf[50];
	
	Debug('t', "StupidExperimenterPlugin::RatConsumedFood");
	numFood--;
	// Take rat off maze when all food gone
	if (numFood <= 0) {
		EXP_TakeRatOffMaze();
		EXP_PlaceRatAt(CENTER); // later, this should not be necessary
		// taking the rat off the maze right now just stops the rat in his tracks
		// and does not reposition the rat at all.
	}
	
	sprintf(buf, "%d", numFood);
	Debug('t', "StupidExperimenterPlugin::RatConsumedFood: numFood= ", buf);
}


